/***************************************************************************
 *
 * Copyright (C) 2001 International Business Machines
 * All rights reserved.
 *
 * This file is part of the GPFS mmfslinux kernel module.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions 
 * are met:
 *
 *  1. Redistributions of source code must retain the above copyright notice, 
 *     this list of conditions and the following disclaimer. 
 *  2. Redistributions in binary form must reproduce the above copyright 
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution. 
 *  3. The name of the author may not be used to endorse or promote products 
 *     derived from this software without specific prior written
 *     permission. 
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR 
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES 
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF 
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *************************************************************************** */
/*
 * Initialize the linux module
 *
 * Contents:
 *
 * $Id: gplInit.c,v 1.21 2001/11/07 16:50:17 dcraft Exp $
 *
 * $Log: gplInit.c,v $
 * Revision 1.21  2001/11/07 16:50:17  dcraft
 * Defect 353738: Manage a ref count on the mmfs module for every mount.
 * An additionaly ref will be on the module as long as the daemon is
 * alive.  We're doing this because if the daemon dies with mount
 * points still active, the mmfs module can be unloaded by rmmod but
 * the code doesn't successfully clean everything up.  Thus the rmmod
 * needs to be blocked from the unload attempt by ref counts.
 *
 * Revision 1.20  2001/09/22 20:08:03  dcraft
 * Remove kiobufs from cxiKernelIODescriptor_t.  Use temporary
 * kiobufs for map/unmap.   Remove dead code and dead comments
 * in portability layer and update readmes and license.
 * Fix traceback to appear in mmfs.log file.
 *
 * Revision 1.19  2001/08/23 22:55:29  manoj
 * Define an interface to allow a machine SID to be registered for access control.
 *
 * Revision 1.18  2001/08/16 21:22:45  manoj
 * Fix problems with notifying oplock breaks with wait queues. Pass openFile as an
 * argument to breakSMBOplock to check for oplock status.
 *
 * Revision 1.17  2001/08/04 00:42:26  tee
 * Remove LINUX_MMAP ifdefs
 *
 * Revision 1.16  2001/07/19 23:24:59  dcraft
 * Modified linux trace to allow non blocking trace record
 * writes (format is TRACE?N).  New gpfs swapd process created
 * which is responsible for reclaiming inodes (5 percent every
 * time it runs).  Marked all our inodes so that they would be
 * ignored by linux kswapd.  Added "unused" inode to inode
 * cache that could be used as a signal that linux kswapd is
 * running and kick off gpfs swapd.  Added means to ignore attempts
 * to kill mmfsd by kswapd if the system gets low on memory.
 * All done in an attempt to avoid kswapd premature wakeup on simple
 * locks and mutexes.
 *
 * Revision 1.15  2001/07/11 03:56:56  manoj
 * Define gpfsCheckNTAccess to check if NT caller has access to a file through
 * individual NT permission bits (other that standard rwx which can be checked by
 * access() call).
 *
 * Revision 1.14  2001/07/10 12:06:26  jpalmer
 * Add function to allow SMB Open and FCNTL tokens to be moved from one system to
 * another in response to the NAS load balancer moving the users.  Add the
 * external interface to provide a lock clamping function that will block new
 * locks during the time NFS is recovering its FCNTL and File locks.
 *
 * Revision 1.13  2001/06/19 17:45:36  eshel
 * Add gpfsSyncNFS to sync attributes for NFS setattr call.
 *
 * Revision 1.12  2001/05/12 18:38:26  schmuck
 * Fix problem with NFS returning spurious errors under load.
 *
 * Revision 1.11  2001/05/08 13:40:35  dixonbp
 * kxRegisterCleanup for linux and gpfs_f_cleanup/gpfsCleanup
 * to do the equivalent of what fopClose does on aix.
 *
 * Revision 1.10  2001/05/01 18:35:39  schmuck
 * Fix cast to avoid compiler warning
 *
 * Revision 1.9  2001/04/29 21:52:32  schmuck
 * SMB_LOCKS code cleanup:
 * In-line openlock state updates on fast open/close path.
 * Remember whether an open (share-mode) lock was obtained;
 * if not, skip cleanup code in close.
 * Removed unused 'isSamba' flag in vinfo.
 *
 * Revision 1.8  2001/04/23 18:09:08  eshel
 * Call DaemonToDie to get internal dump.
 *
 * Revision 1.7  2001/04/21 01:04:24  wyllie
 * Use new infrastructure for allocating and making accessible memory shared
 * between the kernel and the daemon.
 *
 * Revision 1.6  2001/04/19 20:50:19  wyllie
 * Checkpoint partial code to dynamically grow the memory shared between the
 * kernel and the GPFS daemon
 *
 * Revision 1.5  2001/04/11 20:50:45  gjertsen
 * Finish C++ to C conversion of layer.
 *
 * Revision 1.35  2001/04/10 21:10:46  wyllie
 * Convert cxiIOBuffer.C from C++ to C.
 *
 * Revision 1.34  2001/04/03 17:21:37  eshel
 * Add delete_inode() to mark inode for destruction when on last linux iput() call.
 *
 * Revision 1.33  2001/03/28 08:50:15  manoj
 * Support for DOS attributes (NSS only). Define gpfs_ops GetDosAttr and
 * SetDosAttr for the purpose.
 *
 * Revision 1.32  2001/03/16 20:45:28  wsawdon
 * Variety of fixes related to delete snapshot.
 *
 * Revision 1.31  2001/03/13 23:54:45  eshel
 * remove unused code
 *
 * Revision 1.30  2001/03/12 19:11:53  manoj
 * Add kernel oplock support directly into GPFS mmfslinux instead of being a
 * separate module.
 *
 * Revision 1.29  2001/03/09 02:28:28  schmuck
 * Changes for SMB oplocks.
 *
 * Revision 1.28  2001/02/27 01:13:18  eshel
 * add gpfs_ops.gpfsReady()
 *
 * Revision 1.27  2001/02/15 21:57:31  manoj
 * Support for NSS (NAS Security System). The filesystem stores NT-style ACLs and
 * uses NSS for access control and permission checking. All NSS code should be
 * #ifdef'ed by CONFIG_NSS.
 *
 * Revision 1.26  2000/12/15 13:56:41  gjertsen
 * Clean up documentation.
 *
 * Revision 1.25  2000/12/12 20:23:07  gjertsen
 * Make picky IA64 compiler happy in 1002 gnu toolchain.
 *
 * Revision 1.24  2000/12/12 16:34:15  wyllie
 * Remove MemPool operations from gpfs_ops.
 *
 * Revision 1.23  2000/12/01 17:39:04  schmuck
 * Add type casts to keep 64-bit compiler happy.
 *
 * Revision 1.22  2000/12/01 02:10:58  schmuck
 * Instead of assigning NULL function pointers when initializing or resetting the
 * gpfs_operations table, have it point to a dummy function that returns ENOSYS.
 * This avoids having to check for NULL before each call.
 *
 * Revision 1.21  2000/11/06 19:56:12  gjertsen
 * Linux code cleanup and put in build safeguards.
 *
 * Revision 1.20  2000/11/03 20:26:59  dcraft
 * Build SMP, UP, NOHIGHMEM, and 4G memory variations of mmfslinux in
 * one pass.   Names are mmfslinux_UP1G, mmfslinux_UP4G, mmfslinux_SMP1G,
 * mmfslinux_SMP4G.
 *
 * Revision 1.19  2000/11/02 19:46:24  gjertsen
 * Linux code split. Pull out NBD stuff.
 *
// Revision 1.18  2000/10/26  20:52:37  gjertsen
// Purge out ugly USE_CWRAPPERS and export module symbols explicitly
// as the default (in IA64 safe manner).
//
// Revision 1.17  2000/10/24  14:04:47  gjertsen
// Clean up linux module specific code so that gpfs can be
// compiled in the kernel (to allow IA64 kernel debugging).
//
// Revision 1.16  2000/10/20  23:20:14  wyllie
// Use direct calls to functions in gpfs_ops instead of going through a local
// static pointer.
//
// Revision 1.15  2000/08/31  20:35:32  gjertsen
// Various changes to fix IA64 build. Main ones are resolving printf/printk
// headache in gpfslinux.h and making errno global in super.C and exporting it.
//
// Revision 1.14  2000/08/29  18:32:00  dcraft
// Header include cleanup.
//
// Revision 1.13  2000/08/28  14:13:25  gjertsen
// Need to export all kernel symbols explicitly in IA64
// Linux due to bug with insmod.
//
// Revision 1.12  2000/08/03  23:59:03  eshel
// fix trace
//
// Revision 1.11  2000/08/03  23:13:06  eshel
// fix nbd support
//
// Revision 1.10  2000/08/01  17:07:36  wyllie
// Buffer management for Linux, phase II: for each Buffer created by the page
// pool manager, create a shadow of the buffer in the kernel that contains
// kiobufs pointing to the Linux struct page objects for each page in the data
// area of the Buffer.  Use these mappings to implement uXfer, kXfer, etc.
// Not yet fully functional; requires a -D flag to activate.
//
// Revision 1.9  2000/07/31  21:12:27  dcraft
// Define cxiDirent_t and move some defines out of gpfslinux.h
// to cxiTypes.h
//
// Revision 1.8  2000/07/25  16:18:10  gjertsen
// Add in function prototypes and a few type cast fixes.
//
// Revision 1.7  2000/07/21  20:08:10  eshel
// Keep hold on the mmfs (ibm) module while daemon is running.
//
// Revision 1.6  2000/07/20  15:49:21  dcraft
// Abstract event words for OS independence
//
// Revision 1.5  2000/07/19  13:20:11  gjertsen
// Clean up code by adding typecasts, func prototypes, and misc compile fixes.
// Take care of module.h using `new' keyword for IA64.
//
// Revision 1.4  2000/06/12  20:48:37  eshel
// Add mount helper for linux.
//
// Revision 1.3  2000/06/07  16:54:10  eshel
// Change PDEBUG to TRACE
//
// Revision 1.2  2000/05/31  01:06:10  eshel
// final step of module split
//
// Revision 1.1  2000/05/30  23:41:52  eshel
// more for the module split
//
 * Revision 1.2  2000/05/26  00:35:30  eshel
 * more on the way to splitting the kernel code
 *
 * Revision 1.1  2000/05/24  23:43:18  eshel
 * new file to initialize the linux module
 *
 *
 */

#include <Shark-gpl.h>

#include <linux/types.h>
#include <linux/string.h>
#include <linux/version.h>
#include <linux/kernel.h>
#include <linux/module.h>

#include <cxiSystem.h>
#include <cxi2gpfs.h>
#include <linux2gpfs.h>
#include <cxiIOBuffer.h>
#include <cxiSharedSeg.h>
#include <Trace.h>

extern char *prog_path;
extern int trace_level;

struct gpfs_operations gpfs_ops;

#ifdef MODULE

static struct module *ibmModule = 0;
EXTERNC void 
cxiExportModuleStruct(void *modAddr)
{
  ibmModule = (struct module *)modAddr;
}

EXTERNC void 
cxiIncModuleCounter(int up)
{
  TRACE2(TRACE_VNODE, 1, TRCID_CXIINC_COUNTER,
         "cxiIncModuleCounter: ibmModule 0x%X up %d\n",
         ibmModule, up);

  if (ibmModule)
  {
    if (up)                           /* inc if up not = 0 */
      __MOD_INC_USE_COUNT(ibmModule);
    else                              /* dec if up = 0 */
      if (__MOD_IN_USE(ibmModule))
        __MOD_DEC_USE_COUNT(ibmModule);
  }
}

#else /* ! MODULE */

static void *ibmModule = 0;
EXTERNC void cxiExportModuleStruct(void *modAddr) {};
EXTERNC void cxiIncModuleCounter(int up) {};

#endif /* MODULE */

extern char bin_path[CXI_MAXPATHLEN+1];
extern char mmfs_path[CXI_MAXPATHLEN+1];
extern int mmfsd_module_active;


/* Dummy operations for initializing gpfs_operations table.
   Using this dummy operation instead of storing a NULL function pointer
   avoids having to check for NULL before each call. */
EXTERNC void gpfs_void_op()
{
}

EXTERNC int gpfs_enosys_op()
{
  return ENOSYS;
}

EXTERNC void *gpfs_nullptr_op()
{
  return NULL;
}

typedef void   (*VoidFnP)();
typedef int    (*IntFnP)();
typedef void * (*PtrFnP)();

/* initialize gpfs_operations table */
EXTERNC void
reset_gpfs_operations()
{
  gpfs_ops.mmfs = 0;
  * (IntFnP *) & gpfs_ops.gpfsMount = (IntFnP)gpfs_enosys_op;
  * (IntFnP *) & gpfs_ops.gpfsStatfs = (IntFnP)gpfs_enosys_op;
  * (IntFnP *) & gpfs_ops.gpfsSyncfs = (IntFnP)gpfs_enosys_op;
  * (IntFnP *) & gpfs_ops.gpfsFsync = (IntFnP)gpfs_enosys_op;
  * (IntFnP *) & gpfs_ops.gpfsSyncNFS = (IntFnP)gpfs_enosys_op;
  * (IntFnP *) & gpfs_ops.gpfsMkdir = (IntFnP)gpfs_enosys_op;
  * (IntFnP *) & gpfs_ops.gpfsLink = (IntFnP)gpfs_enosys_op;
  * (IntFnP *) & gpfs_ops.gpfsOpen = (IntFnP)gpfs_enosys_op;
  * (IntFnP *) & gpfs_ops.gpfsInodeRead = (IntFnP)gpfs_enosys_op;
  * (IntFnP *) & gpfs_ops.gpfsInodeDelete = (IntFnP)gpfs_enosys_op;
  * (IntFnP *) & gpfs_ops.gpfsRemove = (IntFnP)gpfs_enosys_op;
  * (IntFnP *) & gpfs_ops.gpfsRename = (IntFnP)gpfs_enosys_op;
  * (IntFnP *) & gpfs_ops.gpfsRmdir = (IntFnP)gpfs_enosys_op;
  * (IntFnP *) & gpfs_ops.gpfsSetattr = (IntFnP)gpfs_enosys_op;
  * (IntFnP *) & gpfs_ops.gpfsSymlink = (IntFnP)gpfs_enosys_op;
  * (IntFnP *) & gpfs_ops.gpfsFsyncRange = (IntFnP)gpfs_enosys_op;
  * (IntFnP *) & gpfs_ops.gpfsClose = (IntFnP)gpfs_enosys_op;
  * (IntFnP *) & gpfs_ops.gpfsUnmap = (IntFnP)gpfs_enosys_op;
  * (IntFnP *) & gpfs_ops.gpfsFattr = (IntFnP)gpfs_enosys_op;
  * (IntFnP *) & gpfs_ops.gpfsFclear = (IntFnP)gpfs_enosys_op;
  * (IntFnP *) & gpfs_ops.gpfsFtrunc = (IntFnP)gpfs_enosys_op;
  * (IntFnP *) & gpfs_ops.gpfsRead = (IntFnP)gpfs_enosys_op;
  * (IntFnP *) & gpfs_ops.gpfsWrite = (IntFnP)gpfs_enosys_op;
  * (IntFnP *) & gpfs_ops.gpfsGetattr = (IntFnP)gpfs_enosys_op;
  * (IntFnP *) & gpfs_ops.gpfsAccess = (IntFnP)gpfs_enosys_op;
  * (IntFnP *) & gpfs_ops.gpfsReaddir = (IntFnP)gpfs_enosys_op;
  * (IntFnP *) & gpfs_ops.gpfsReadlink = (IntFnP)gpfs_enosys_op;
  * (IntFnP *) & gpfs_ops.gpfsCreate = (IntFnP)gpfs_enosys_op;
  * (IntFnP *) & gpfs_ops.gpfsMknod = (IntFnP)gpfs_enosys_op;
  * (IntFnP *) & gpfs_ops.gpfsRele = (IntFnP)gpfs_enosys_op;
  * (IntFnP *) & gpfs_ops.gpfsLookup = (IntFnP)gpfs_enosys_op;
  * (IntFnP *) & gpfs_ops.gpfsFcntl = (IntFnP)gpfs_enosys_op;
  * (VoidFnP*) & gpfs_ops.gpfsUncache = (VoidFnP)gpfs_void_op;
  * (IntFnP *) & gpfs_ops.gpfsUnmount = (IntFnP)gpfs_enosys_op;
  * (VoidFnP*) & gpfs_ops.gpfsFinishUnmount = (VoidFnP)gpfs_void_op;
  * (IntFnP *) & gpfs_ops.gpfsFcntlReset = (IntFnP)gpfs_enosys_op;
  * (IntFnP *) & gpfs_ops.gpfsGetAcl = (IntFnP)gpfs_enosys_op;
  * (IntFnP *) & gpfs_ops.gpfsPutAcl = (IntFnP)gpfs_enosys_op;
#ifdef CONFIG_NSS
  * (IntFnP *) & gpfs_ops.gpfsGetDosAttr = (IntFnP)gpfs_enosys_op;
  * (IntFnP *) & gpfs_ops.gpfsPutDosAttr = (IntFnP)gpfs_enosys_op;
  * (IntFnP *) & gpfs_ops.gpfsCheckNTAccess = (IntFnP)gpfs_enosys_op;
  * (IntFnP *) & gpfs_ops.gpfsSetMachineSID = (IntFnP)gpfs_enosys_op;
#endif
  * (IntFnP *) & gpfs_ops.gpfsGetNFS = (IntFnP)gpfs_enosys_op;
  * (IntFnP *) & gpfs_ops.gpfsReleaseNFS = (IntFnP)gpfs_enosys_op;
  * (IntFnP *) & gpfs_ops.gpfsReady = (IntFnP)gpfs_enosys_op;
  * (IntFnP *) & gpfs_ops.gpfsMmap = (IntFnP)gpfs_enosys_op;
  * (IntFnP *) & gpfs_ops.pagerKprocMainThread = (IntFnP)gpfs_enosys_op;
#ifdef SMB_LOCKS
  * (IntFnP *) & gpfs_ops.SMBOpenLockControl = (IntFnP)gpfs_enosys_op;
  * (IntFnP *) & gpfs_ops.SMBGetOplockState = (IntFnP)gpfs_enosys_op;
  * (IntFnP *) & gpfs_ops.gpfsSetSMBOplock = (IntFnP)gpfs_enosys_op;
  * (IntFnP *) & gpfs_ops.gpfsClampLocks = (IntFnP)gpfs_enosys_op;
#endif
  * (VoidFnP*) & gpfs_ops.gpfsDaemonToDie = (VoidFnP)gpfs_enosys_op;
  * (IntFnP *) & gpfs_ops.gpfsCleanup = (IntFnP)gpfs_enosys_op;
  * (VoidFnP*) & gpfs_ops.gpfsSwapdSignal = (VoidFnP)gpfs_enosys_op;
};

EXTERNC  void gpfs_clean()
{
  kill_mmfsd();
  if (gpfs_ops.mmfs)
  {
    gpfs_ops.mmfs(CFG_TERM);
  }
  gpfs_unreg_fs();
  gpfs_block_clean();
}

EXTERNC  int gpfs_init()
{
  if (strlen(prog_path) > CXI_MAXPATHLEN)
  {
    TRACE2(TRACE_SSEG, 0, TRCID_GPFSINIT_000,
         "gpfs_init: prog_path %s length %d too long\n",
         prog_path, strlen(prog_path));
    return -1;
  }
  if (gpfs_block_init())
    return -1;
  if (gpfs_reg_fs())
  {
    gpfs_block_clean();
    return -1;
  }
  if (gpfs_ops.mmfs)
  {
    if (gpfs_ops.mmfs(CFG_INIT))
    {
      gpfs_clean();
      return -1;
    }
  }
  else
    return -1;

  strcpy(bin_path, prog_path);
  return 0;
}

/* Module initialization */
EXTERNC
#ifdef MODULE
int init_module(void)
#else
int portability_init(void)
#endif /* MODULE */
{
  int i;

  for (i=0 ; i<MAX_TRACE_CLASSES ; i++)
     TraceFlagsP[i] = 9;

  InitSharedKernelMemory();

  reset_gpfs_operations();

  TRACE2(TRACE_SSEG, 0, TRCID_PORTINIT_001,
         "init_module: prog_path %s trace level %d\n",
         prog_path, trace_level);

  /* Ensure types are the correct size */
  if (cxiCheckTypes())
    return -1;

  if (ss_init())
    return -1;

#ifdef CONFIG_NSS
  cred_init();
#endif

#ifdef SMB_LOCKS
  oplock_init();
#endif

  gpfs_proc_export_init();

  KibdModuleInit();

  return 0;
}

/* Module unload */
EXTERNC
#ifdef MODULE
void cleanup_module(void)
#else
void portability_cleanup(void)
#endif /* MODULE */
{
#ifdef SMB_LOCKS
  oplock_cleanup();
#endif

#ifdef CONFIG_NSS
  cred_cleanup();
#endif

  gpfs_proc_export_term();

  KibdModuleTerm();

  TermSharedKernelMemory();

}
